<?php
/**
 * AgenWebsite Shipping - Tracking Module
 *
 * @author AgenWebsite Shipping
 * @package AgenWebsite Shipping
 * @since 1.0.0
 */

use Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController;

if (!defined('ABSPATH')) {
    exit;
} // Exit if accessed directly

if (!class_exists('AWShipping_Tracking')) :

    class AWShipping_Tracking
    {
        private $tracking_meta = '_woocommerce_agenwebsite_tracking_number';
        private $tracking_courier_meta = '_woocommerce_agenwebsite_tracking_courier';
        private $tracking_nonce = 'woocommerce_agenwebsite_tracking__nonce';
        private $tracking_action = 'woocommerce_agenwebsite_tracking_field';

        public function __construct()
        {
            add_action('add_meta_boxes', array(&$this, 'add_meta_box'));
            add_action('woocommerce_process_shop_order_meta', array($this, 'save'), 1);

            add_action('init', array(&$this, 'register_shipped_order_status'));
            add_filter('wc_order_statuses', array(&$this, 'add_shipped_status'));

            add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts'], 10);
            add_action('woocommerce_view_order', array(&$this, 'view_order'));
        }

        public function add_meta_box($post_type)
        {
            $screen = class_exists( '\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController' ) && wc_get_container()->get( CustomOrdersTableController::class )->custom_orders_table_usage_is_enabled()
                ? wc_get_page_screen_id( 'shop-order' )
                : 'shop_order';

            add_meta_box(
                'agenwebsite-shipping-tracking',
                __('AgenWebsite Shipping Tracking', 'agenwebsite-shipping'),
                array(&$this, 'render_meta_box'),
                $screen,
                'side',
                'high'
            );

            return true;
        }

        public function save($post_id)
        {
            if (!isset($_POST[$this->tracking_nonce]))
                return $post_id;

            $tracking_nonce = $_POST[$this->tracking_nonce];

            if (!wp_verify_nonce($tracking_nonce, $this->tracking_action))
                return $post_id;

            if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE)
                return $post_id;

            if ('page' == $_POST['post_type']) {
                if (!current_user_can('edit_page', $post_id))
                    return $post_id;
            } else {
                if (!current_user_can('edit_post', $post_id))
                    return $post_id;
            }

            $input = sanitize_text_field($_POST['agenwebsite_awb_number']);
            update_post_meta($post_id, $this->tracking_meta, $input);

            $inputTrackingCourier = sanitize_text_field($_POST['tracking_courier_name']);
            update_post_meta($post_id, $this->tracking_courier_meta, $inputTrackingCourier);

            return true;
        }

        public function render_meta_box($post_id)
        {
            $order = wc_get_order($post_id);

            $order_shipping_methods = $order->get_shipping_methods();
            $shipping_method = @array_shift($order_shipping_methods);
            $courier = $shipping_method->get_meta('courier');

            $selected_courier = get_post_meta($post_id, $this->tracking_courier_meta, true);
            $selected_courier = !empty($selected_courier) ? $selected_courier : $courier;

            $options = aw_shipping_options();
            $master_courier = aw_shipping_data('couriers');
            $list_courier = $options['courier'];

            // nonce field for the agenwebsite tracking
            wp_nonce_field($this->tracking_action, $this->tracking_nonce);

            $value_tracking = get_post_meta($post_id, $this->tracking_meta, TRUE);

            // Start Wrapper
            echo '<div class="agenwebsite-shipping_wrapper">' . "\n";

            echo '<div class="form-wrap"><div class="form-field">' . "\n";
            // Create meta box content
            echo '<label for="agenwebsite_awb_number">' . __('Nomor Resi', 'agenwebsite-shipping') . '</label>' . "\n";
            echo '<input type="text" name="agenwebsite_awb_number" id="agenwebsite_awb_number" value="' . esc_attr($value_tracking) . '" placeholder="Masukkan nomor resi" />' . "\n";

            echo '</div></div>' . "\n";

            echo '<div class="form-wrap"><div class="form-field">' . "\n";
            // Create meta box content
            echo '<label for="tracking_courier_name">Kurir</label>' . "\n";
            echo '<select name="tracking_courier_name" id="tracking_courier_name" style="width: 100%">';
            foreach ($master_courier as $courier) {
                if (!in_array($courier['id'], $list_courier)) continue;
                echo '<option value="' . $courier['id'] . '" ' . selected($courier['id'], $selected_courier, true) . '>' . $courier['title'] . '</option>';
            }
            echo '</select>';


            echo '</div></div>' . "\n";

            do_action('woocommerce_agenwebsite_tracking_meta_box');
            // End Wrapper
            echo '</div>' . "\n";
        }

        public function register_shipped_order_status()
        {
            register_post_status('wc-shipped', array(
                'label' => 'Shipped',
                'public' => true,
                'exclude_from_search' => false,
                'show_in_admin_all_list' => true,
                'show_in_admin_status_list' => true,
                'label_count' => _n_noop('Shipped <span class="count">(%s)</span>', 'Shipped <span class="count">(%s)</span>')
            ));
        }

        public function add_shipped_status($order_statuses)
        {
            $new_order_statuses = array();
            // add new order status after on-hold
            foreach ($order_statuses as $key => $status) {
                $new_order_statuses[$key] = $status;
                if ('wc-on-hold' === $key) {
                    $new_order_statuses['wc-shipped'] = 'Shipped';
                }
            }
            return $new_order_statuses;
        }

        public function enqueue_scripts()
        {
            if (is_view_order_page()) {
                wp_enqueue_style('aw-shipping-tracking', aw_shipping_plugin_url() . '/assets/css/agenwebsite-shipping-tracking.css', array('select2'), '1.0.0', 'all');
                wp_enqueue_script('aw-shipping-tracking', aw_shipping_plugin_url() . '/assets/js/agenwebsite-shipping-tracking.js', array('jquery', 'select2'), '1.0.0', true);

                $localize = [
                    'ajax_url' => admin_url('admin-ajax.php'),
                    '_wpnonce' => wp_create_nonce('aw_shipping_nonce'),
                ];
                wp_localize_script('aw-shipping-tracking', 'aw_shipping', $localize);
            }
        }

        public function view_order($order_id)
        {
            $this->display_tracking_info($order_id, true);
        }

        public function display_tracking_info($orderid, $show_tracking = false, $to_email = false)
        {
            $tracking_number = get_post_meta($orderid, $this->tracking_meta, TRUE);

            // check if tracking number is empty
            if (!$tracking_number) return;

            $tracking_option = aw_shipping_options();

            $message = $tracking_option['tracking_message'];

            $order = wc_get_order($orderid);
            $tracking_name = $order->get_shipping_method();
            $order_shipping_methods = $order->get_shipping_methods();
            $shipping_method = @array_shift($order_shipping_methods);
            $courier = $shipping_method->get_meta('courier');

            $selected_courier = get_post_meta($orderid, $this->tracking_courier_meta, true);
            $selected_courier = !empty($selected_courier) ? $selected_courier : $courier;

            $shortcode = array(
                '[agenwebsite_tracking_title]' => $tracking_name,
                '[agenwebsite_tracking_number]' => $tracking_number
            );
            echo '<div class="tracking-order-">';
            echo wpautop(str_replace(array_keys($shortcode), array_values($shortcode), $message));
            if ($show_tracking) {
                $button = $tracking_option['tracking_button'] ?? 'Lacak Pesanan';
                $result = '';
                $class = 'view-order-tracking';

                ob_start();
                wc_get_template('tracking/agenwebsite-tracking.php', array(
                    'class' => $class,
                    'order_id' => $orderid,
                    'noresi' => $tracking_number,
                    'courier' => $selected_courier,
                    'button' => $button,
                    'result' => $result,
                    'checkout_url' => $order->get_view_order_url()
                ), sprintf('agenwebsite-shipping'), untrailingslashit(aw_shipping_plugin_path()) . '/templates/');

                $output = ob_get_clean();
                echo $output;
            }

            if ($to_email) {
                $button = __('Lacak Resi', 'agenwebsite-shipping');
                ob_start();
                wc_get_template('tracking/agenwebsite-tracking-button.php', array(
                    'button' => $button,
                    'checkout_url' => $order->get_view_order_url()
                ), sprintf('agenwebsite-shipping'), untrailingslashit(aw_shipping_plugin_path()) . '/templates/');

                $output = ob_get_clean();
                echo $output;
            }
            echo '</div>';
        }
    }

    new AWShipping_Tracking();

endif;