<?php
/**
 * AgenWebsite Shipping API
 *
 * @author AgenWebsite Shipping
 * @package AgenWebsite Shipping
 * @since 1.0.0
 */
if (!defined('ABSPATH')) {
    exit;
} // Exit if accessed directly

if (!class_exists('AWShipping_Api')):

    class AWShipping_Api
    {

        /* @var array */
        private $data;

        /**
         * Constructor
         *
         * @return void
         * @since 1.0.0
         */
        public function __construct()
        {
            $this->product = 'agenwebsite-shipping';
            $this->product_version = '2.0.0';
            $this->license_code = aw_shipping_license();
            $this->url = 'https://api-v2.agenwebsite.com/v2';
            $this->timeout = apply_filters('woocommerce_agenwebsite_api_timeout', 20);
        }

        /**
         * Magic Setter
         *
         * @return void
         * @since 1.0.0
         */
        public function __set($name, $value)
        {
            $this->data[$name] = $value;
        }

        /**
         * Magic Getter
         *
         * @return mixed
         * @since 1.0.0
         */
        public function __get($name)
        {
            if (array_key_exists($name, $this->data)) {
                return $this->data[$name];
            }

            return false;
        }

        /**
         * Get URL of API AgenWebsite
         *
         * @access public
         * @param string $method
         * @param array $param
         * @return string $uri
         * @since 1.0.0
         */
        public function get_uri($method, $param = array())
        {

            $uri = $this->url;
            $this->method = $method;

            switch ($method) {
                case 'license':
                    $uri .= '/license/activate';
                    break;
                case 'search_data':
                    $uri .= '/shipping/data';
                    break;
                case 'price':
                    $uri .= '/shipping/price';
                    break;
                case 'price-international':
                    $uri .= '/shipping/price-international';
                    break;
                case 'tracking':
                    $uri .= '/shipping/tracking';
                    break;
            }

            $uri .= $this->build_param($param);

            return $uri;
        }

        /**
         * Build param API from array
         *
         * @access private
         * @param array $param
         * @return string $param
         * @since 1.0.0
         */
        private function build_param($args = array())
        {
            $param = '';

            $i = 0;
            foreach ($args as $name => $value) {
                $param .= ($i == 0) ? '?' : '';
                $param .= $name . '=' . rawurlencode($value);
                $param .= (++$i == count($args)) ? '' : '&';
            }

            return $param;
        }

        /**
         * Check license empty or not
         *
         * @access public
         * @return bool
         * @since 1.0.0
         */
        public function check_license()
        {
            if ($this->license_code == '') {
                $result['status'] = 'error';
                $result['message'] = __('Kode Lisensi belum diisi.', 'agenwebsite-shipping');
                $result['result'] = '';

                $this->result = $result;

                return false;
            }

            return true;
        }

        /**
         * Remote get
         *
         * @access public
         * @return array $result
         * @since 1.0.0
         */
        public function remote_get($method, $param = array(), $force = false)
        {

            if ($this->check_license() || $force) {

                $param_uri = $this->build_param($param);

                $param['license'] = $this->license_code;
                $param['product'] = $this->product;
                $param['currency'] = get_woocommerce_currency();
                $param['converter'] = 'google';

                $headers['browser'] = sprintf('%s; %s', $this->browser['name'], $this->browser['version']);
                $headers['platform'] = $this->browser['platform'];
                $headers['app'] = sprintf('%ss; %s', $this->product, $this->product_version);
                $headers['param_uri'] = $param_uri;

                $this->response = wp_remote_get($this->get_uri($method, $param), array('timeout' => $this->timeout, 'headers' => $headers));

                $this->process_result();

            }

            return $this->result;

        }

        /**
         * Remote put
         *
         * @access public
         * @return array $result
         * @since 1.0.0
         */
        public function remote_post($method, $param = array(), $additional = array())
        {
            $body['license'] = $this->license_code;
            $body['product'] = $this->product;
            $body = array_merge($body, $additional);

            $this->response = wp_remote_post($this->get_uri($method, $param), array('timeout' => $this->timeout, 'body' => $body));

            $this->process_result();

            return $this->result;
        }

        /**
         * Process result
         *
         * @access public
         * @return void
         * @since 1.0.0
         */
        public function process_result()
        {
            $cant_connect = false;

            if (!is_wp_error($this->response)) {
                $body = json_decode($this->response['body'], TRUE);

                $code = wp_remote_retrieve_response_code($this->response);
                if ($code == 200) {
                    $result['status'] = 'success';
                    $result['message'] = $body['message'];
                    $result['result'] = $body['data'];
                } else {
                    $result['status'] = 'error';
                    $result['message'] = $body['message'] ?? 'Gagal terhubung dengan AgenWebsite';
                    $result['result'] = '';
                }

            } else {

                $cant_connect = true;

            }

            if ($cant_connect) {
                $result['status'] = 'error';
                $result['message'] = __('Gagal terhubung dengan AgenWebsite', 'agenwebsite-shipping');
                $result['result'] = '';
            }

            $this->result = $result;

        }
    }

endif;
